﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using PalmSens;

namespace PalmSens
{
    /// <summary>
    /// The code for communication with the EmStat Pico bootloader
    /// </summary>
    internal class PSBootloaderEnc
    {
        enum Command
        {
            //Available in both bootloader and normal mode
            CMD_VERSION = 't',
            CMD_RESET = 'b',

            //Bootloader only cmds
            CMD_UPLOAD_FW_START = 's',
            CMD_UPLOAD_FW_DATA = 'd',
            CMD_UPLOAD_FW_END = 'e',
        };


        private const int BLOCK_SIZE = 128;
        private StreamDevice _device;
        private bool _inited = false;

        internal PSBootloaderEnc(StreamDevice streamDevice)
        {
            _device = streamDevice;
        }

        internal void Init()
        {
            //Clear buff
            _device.SendCmd("");

            string versionString = _device.SendCmd(((char)Command.CMD_VERSION).ToString());
            if (versionString.StartsWith("espico") || versionString.StartsWith("senswb") || versionString.StartsWith("esp2__"))
            {
                //Device not in bootloader mode, attempt to enter bootloader.
                _device.SendCmd("dlfw");
                System.Threading.Thread.Sleep(200);
                versionString = _device.SendCmd(((char)Command.CMD_VERSION).ToString());
            }
            if (!versionString.StartsWith("espbl") && !versionString.StartsWith("senwbl") && !versionString.StartsWith("esp2bl"))
                throw new IOException("Could not communicate with device");

            _inited = true;
        }

        /// <summary>
        /// returns 16 bits where bit 0-7: sum1 8-15: sum2
        /// </summary>
        private UInt16 Fletcher16(byte[] data, int startIndex, int count)
        {
            UInt16 sum1 = 0;
            UInt16 sum2 = 0;
            int index;

            for (index = startIndex; index < startIndex+count; ++index)
            {
                sum1 = (UInt16)((sum1 + (UInt16)data[index]) % 255);
                sum2 = (UInt16)((sum2 + sum1) % 255);
            }

            return (UInt16)((sum2 << 8) | sum1);
        }

        internal void UploadFile(byte[] fwFile)
        {
            string resp;

            if (!_inited)
                throw new InvalidOperationException("Not initialized");

            //Signal start of fw file
            resp = _device.SendCmd("startfw");
            if (resp != "\n")
                throw new IOException($"Unexpected response from \"startfw\": {resp}");

            //Send blocks
            for (int i = 0; i < fwFile.Length; i += BLOCK_SIZE)
            {
                int ntries = 5;
                while (ntries > 0)
                {
                    try
                    {
                        byte nbytes = (byte)((i + BLOCK_SIZE > fwFile.Length) ? fwFile.Length - i : BLOCK_SIZE);
                        UInt16 checksum = Fletcher16(fwFile, i, nbytes); //Checksum to verify integrity of package
                        string cmd = "data" + nbytes.ToString("X2") + StreamDevice.ToHex(fwFile, i, nbytes) + checksum.ToString("X4");
                        resp = _device.SendCmd(cmd);
                        //An error will be thrown by the ESPico when: a flash error has occured, the checksum was incorrect or the command was not understood/accepted 
                        if (resp != "\n")
                            throw new IOException($"Unexpected response from \"data\": {resp}");

                        ntries = 0;
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine($"Failed to write block {i}, tries left: {ntries-1}, exception: {ex.Message}");

                        //TODO: Should check for error code here, only comm errors should be retried. Retrying the data package when a flash error has occured is not useful.
                        if (--ntries == 0)
                            throw ex;
                    }
                }
            }

            //Signal end of fw file
            resp = _device.SendCmd("endfw");
            if (resp != "\n")
                throw new IOException($"Unexpected response from \"endfw\": {resp}");

            //Reset device
            resp = _device.SendCmd("boot");
            if (resp != "\n")
                throw new IOException($"Unexpected response from \"boot\": {resp}");
        }
    }
}
